/*************************************************/
/* Examples Program For "ET-dsPIC33WEB-V1" Board */
/* Hardware    : ET-dsPIC33WEB-V1.0              */
/* Target MCU  : dsPIC33FJ128GP708               */
/*       	   : X-TAL : 8.00 MHz                */
/* Config Fosc : XTAL = 8MHz (XT+PLL)            */
/*             : PLLPRE[4..0] = 00000 (N1=2)     */
/*             : VCO Input = 8MHz / 2 = 4MHz     */
/*             : PLLDIV[8..0] = 0x26(M=40)       */
/*             : VCO Output = 4 x 40MHz = 160MHz */
/*             : PLLPOST[1:0] = 0:0 (N2=2)       */
/*             : Fosc = 160MHz / 2 = 80MHz       */
/*             : Fcy = Fosc/2 = 80MHz /2 = 40MHz */
/* Compiler    : MPLAB + C30 V3.01  		     */
/* Write By    : Eakachai Makarn(ETT CO.,LTD.)   */
/* Last Update : 16/August/2007                  */
/* Function    : Example Use I2C Connect PCF8574 */
/*             : LED Moving on Output of PCF8574 */	
/*************************************************/
/* Used ET-MINI I/O PCF8574A Module Interface    */
/* SCL1 = RG2(I2C-SCL)                           */
/* SDA1 = RG3(I2C-SDA)                           */
/* Setup PCF8574A Address = "000"                */
/*************************************************/

#include "p33FJ128GP708.h"                             				// dsPIC33FJ128GP708 MPU Register

/* Setup Configuration For ET-dsPIC33WEB-V1 */
_FOSCSEL(FNOSC_PRIPLL & 											// Select Primary OSC = XT + PLL
         IESO_OFF )													// Startup With User Select OSC
_FOSC(FCKSM_CSDCMD &        										// Clock Switch Disabled	
      OSCIOFNC_OFF &												// OSC2 Pin = Clock O/P
      POSCMD_XT)													// Primary OSC = XT 
_FWDT(FWDTEN_OFF &           										// Disable Watchdog Timer 
      WINDIS_OFF &          										// Disabled Windowed WDT
      WDTPRE_PR128 &        										// Watchdog prescaler=1:128
      WDTPOST_PS32768)     											// Watchdog postscaler=1:32,768
_FPOR(FPWRT_PWR128)       											// Power-on Reset Value = 128ms
_CONFIG3(JTAGEN_OFF);												// Disable JTAG Interface
/* End of Configuration For ET-dsPIC33WEB-V1 */

void delay(unsigned long int);										// Delay Time Function

int main(void)
{  
  unsigned char LED; 												// LED Buffer Display

  /* Start of PLL Config Fcy = 40MIPS */
  CLKDIV &= 0xFFE0;													// PLLPRE[4..0] = 00000 (N1=2)
  CLKDIV &= 0xFF3F;													// PLLPOST[1:0] = 00 (N2=2)
  PLLFBD  = 0x26;													// PLLDIV[8..0] = 000100110(M=40)
  while(!OSCCONbits.LOCK);											// Wait PLL to Lock
  OSCCONbits.CLKLOCK = 1;											// Disable PLL Modify
  /* End of PLL Config Fly = 40MIPS */

  /* Start of Config I2C */
  I2C1CONbits.I2CEN = 0;											// Disable Before Change Config
  I2C1CONbits.I2CSIDL = 1;											// Disable in Idle Mode
  I2C1CONbits.SCLREL = 0;											// I2C Clock Hold
  I2C1CONbits.IPMIEN = 0;   										// Disable I2C IPMI Mode Control
  I2C1CONbits.A10M = 0;  											// I2C Device Address = 7 Bit
  I2C1CONbits.DISSLW = 1;  											// Disable I2C Slew Rate Control
  I2C1CONbits.SMEN = 0;  											// Disable I2C SMBUS Mode
  I2C1CONbits.GCEN = 0; 											// Disable I2C General Call(Slave) 
  I2C1CONbits.STREN = 0;											// Disable SCL Clock Stretch
  I2C1CONbits.ACKDT = 0;											// ACK Cycle = ACK
  I2C1CONbits.ACKEN = 0;											// Disable I2C Acknowledge
  I2C1CONbits.RCEN = 0;												// Disable I2C Receive
  I2C1CONbits.PEN = 0;												// Disable I2C Stop	
  I2C1CONbits.RSEN = 0;												// Disable I2C Restart
  I2C1CONbits.SEN = 0;												// Disable I2C Start
 
  // Setup I2C Baudrate = 100KHz 
  // ET-dsPIC33WEB-V1 Hardware Board
  // Fcy = 40MHz
  // I2CBRG = [(Fcy/Fscl)-(Fcy/1000000)]-1
  //        = [(40 MHz / 100KHz) - (40 MHz / 1000000)] - 1
  //        = [(400)-(40)]-1
  //        = 360 - 1
  //        = 359
  I2C1BRG = 359;													// I2C Baudrate(100 KHz)

  // I2C[1] Interrupt Control 
  IEC1bits.MI2C1IE = 0;												// Disable I2C1 Master Interrupt
  IEC1bits.SI2C1IE = 0;												// Disable I2C1 Slave Interrupt

  I2C1CONbits.I2CEN = 1;											// Enable I2C1 Function
  /* End of Config I2C */

  // Start Test Output I2C(PCF8574A) //
  while(1)															// Loop Continue
  {
    for (LED = 0x01; LED != 0x00; LED <<= 1)						// Shift Left LED(Right <- Left)
    { 
      I2C1CONbits.SEN = 1;											// Send Start Condition
	  while(I2C1CONbits.SEN);										// Wait Start Complete
     
      I2C1TRN = 0x70;												// Write PCF8574A ID Code (0111000+W)
	  while(I2C1STATbits.TBF);										// Wait Write Address Complete

	  while(I2C1STATbits.ACKSTAT);									// Wait Acknowledge Complete

      while(I2C1CONbits.SEN || 										// Wait All I2C Status Clear(Wait Idle) 
            I2C1CONbits.PEN || 
            I2C1CONbits.RCEN || 
            I2C1CONbits.ACKEN || 
            I2C1STATbits.TRSTAT);
     
      I2C1TRN = 0xFF-LED;											// Write Output Data to PCF8574A(Toggle Logic)
      while(I2C1STATbits.TBF);										// Wait Write Data Complete

	  while(I2C1STATbits.ACKSTAT);									// Wait Acknowledge Complete

      while(I2C1CONbits.SEN || 										// Wait All I2C Status Clear (Wait Idle)
            I2C1CONbits.PEN || 
            I2C1CONbits.RCEN || 
            I2C1CONbits.ACKEN || 
            I2C1STATbits.TRSTAT);

      I2C1CONbits.PEN = 1;											// Send Stop Condition
      while(I2C1CONbits.PEN);										// Wait Stop Complete	
      
      delay(1500000);	    										// Display Delay	    			
    }  
  }
}

/***********************/
/* Delay Time Function */
/*    1-4294967296     */
/***********************/
void delay(unsigned long int count1)
{
  while(count1 > 0) {count1--;}										// Loop Decrease Counter	
}


